namespace Loxx
{
  /// <summary>
  /// Type of the graph to be shown. 
  /// Currently the value is only used for the decision &quot;MultiBars&quot; or &quot;SplitBar&quot; -- the other values do not
  /// change the output.
  /// </summary>
  public enum GraphType
  {
    /// <summary>
    /// A XY-graph will be displayed for the data.
    /// </summary>
    XYGraph = 1,
    /// <summary>
    /// A simple bar chart will be displayed for the data.
    /// </summary>
    BarChart = 2,
    /// <summary>
    /// A Multi-bar-chart will be displayed for the data.
    /// </summary>
    MultiBarChart = 3,
    /// <summary>
    /// A split-bar chart will be displayed for the data.
    /// </summary>
    SplitBarChart = 4,
  }
  /// <summary>
  /// The type of the search to be performed (plain text, regex etc.).
  /// </summary>
  public enum SearchType
  {
    /// <summary>
    /// Search for plain text.
    /// </summary>
    PlainText = 0,
    /// <summary>
    /// Search for regular expression matches.
    /// </summary>
    Regex = 1,
    /// <summary>
    /// Search for matches using the standard Windows wildcards ('*' and '?').
    /// </summary>
    Wildcard = 2,
    /// <summary>
    /// Use &quot;intern patterns&quot; for the search. This type of search is able to extract values but is much faster than regular
    /// expressions. 
    /// </summary>
    InternPatterns = 3,
  }
  /// <summary>
  /// Enumeration defining the orientation o a graphic output.(Currently only vertical bar charts are supported).
  /// </summary>
  public enum Orientation
  {
    /// <summary>
    /// Graphic orientation will be vertical.
    /// </summary>
    Vertical = 1,
    /// <summary>
    /// Graphic orientation will be horizontal.
    /// </summary>
    Horizontal = 2,
  }
  /// <summary>
  /// Enum defining what shall be collected during find actions.
  /// </summary>
  public enum CollectTarget
  {
    /// <summary>
    /// No values will be collected.
    /// </summary>
    Nothing = 0,
    /// <summary>
    /// The search will collect string values. The data will be available through the &quot;Match&quot; element in the FindMatchInfo.
    /// </summary>
    Strings = 1,
    /// <summary>
    /// The search will collect values as integer values (the FindMatchInfo will contain in the &quot;IntValue&quot; attribute the
    /// value that matched expecting int values). 
    /// </summary>
    Integer = 2,
    /// <summary>
    /// The search will collect values as hex-integer values (the FindMatchInfo will contain in the &quot;IntValue&quot; attribute
    /// the value that matched expecting hex values). So if a &quot;FFFF&quot; was matching. the result is 65565.
    /// </summary>
    HexInteger = 3,
    /// <summary>
    /// The search will collect double values (with dot as decimal point).
    /// </summary>
    Double = 4,
  }
  /// <summary>
  /// Enum defining the message type for output coloring.
  /// </summary>
  public enum MessageType
  {
    /// <summary>
    /// Use standard output color.
    /// </summary>
    Standard = 0,
    /// <summary>
    /// Use &quot;good&quot; output color.
    /// </summary>
    Success = 1,
    /// <summary>
    /// Use &quot;warning&quot; output color.
    /// </summary>
    Warning = 2,
    /// <summary>
    /// Use &quot;warning&quot; output color.
    /// </summary>
    Error = 3,
  }
  /// <summary>
  /// Alignment for output values.
  /// </summary>
  public enum Alignment
  {
    /// <summary>
    /// Left aligned output.
    /// </summary>
    Left = 0,
    /// <summary>
    /// Centered output.
    /// </summary>
    Center = 1,
    /// <summary>
    /// Right aligned output.
    /// </summary>
    Right = 2,
  }
  /// <summary>
  /// Mode defining how windows shall be arranged.
  /// </summary>
  public enum ArrangeMode
  {
    /// <summary>
    /// Arrange windows as grid.
    /// </summary>
    Grid = 0,
    /// <summary>
    /// Arrange windows in a single tab.
    /// </summary>
    SingleTab = 1,
    /// <summary>
    /// Arrange windows side by side(vertical tiles).
    /// </summary>
    VerticalTiles = 2,
    /// <summary>
    /// Arrange windows like a stack(horizontal tiles).
    /// </summary>
    HorizontalTiles = 3,
    /// <summary>
    /// Put the active view to the left and other windows in a grid on the right.
    /// </summary>
    LeftAndGrid = 4,
    /// <summary>
    /// Put the active view to the top and other windows in a grid on the bottom.
    /// </summary>
    TopAndGrid = 5,
    LeftAndTabbed = 6,
    TopAndTabbed = 7,
  }
  /// <summary>
  /// Alignment enum for moving views. 
  /// </summary>
  public enum DockMode
  {
    /// <summary>
    /// Centered output.
    /// </summary>
    Tabbed = 0,
    /// <summary>
    /// Move view to the left position (splits target and moves the view to the left).
    /// </summary>
    DockLeft = 1,
    /// <summary>
    /// Move view to the right position (splits target and moves the view to the right).
    /// </summary>
    DockRight = 2,
    /// <summary>
    /// Move view to the top position (splits target and moves the view to the top).
    /// </summary>
    DockTop = 3,
    /// <summary>
    /// Move view to the bottom position in the target tab (splits target and moves the view to the bottom).
    /// </summary>
    DockBottom = 4,
  }
  /// <summary>
  /// Loglevel value type.
  /// </summary>
  public enum LogLevel
  {
    None = 0,
    /// <summary>
    /// Log errors only.
    /// </summary>
    Error = 1,
    /// <summary>
    /// Leg errors and warnings.
    /// </summary>
    Warning = 2,
    /// <summary>
    /// Log everything.
    /// </summary>
    Info = 3,
  }
  
  /// <summary>
  /// The Application class represents the application itself (so you find some essential functions like &quot;Quit&quot;).
  /// If using VBA, application class is the default target for all methods, so you can simply write &quot;Quit&quot; or &quot;Application.Quit&quot;
  /// -- the result is the same.
  /// In case of C# macros the Application instance is provided in the execute method as parameter.
  /// </summary>
  public interface Application
  {
    /// <summary>
    /// The current active view. It returns an error if no view is open.
    /// </summary>
    View ActiveView { set;get;}
    
    /// <summary>
    /// Access to the clipboard. Writing the property copies the given string to the windows clipboard, reading returns the current
    /// content.It is a convenience property allowing easy access to the clipboard (can be also done with system calls).
    /// </summary>
    string Clipboard { set;get;}
    
    /// <summary>
    /// Get the current date as string (YYYY-MM-DD).
    /// </summary>
    string DateString { get;}
    
    /// <summary>
    /// Returns an array of match infos after a find in files run.
    /// </summary>
    FindInFileMatchInfo[] FindInFileMatchInfos { get;}
    
    /// <summary>
    /// Returns the number of &quot;find in files&quot;-matches.
    /// </summary>
    int FindInFileMatches { get;}
    
    /// <summary>
    /// Collection of the files with matching &quot;find in files&quot; results.
    /// </summary>
    string[] FindInFilesFiles { get;}
    
    /// <summary>
    /// Get the macro output content as string.
    /// </summary>
    string MacroOutput { get;}
    
    /// <summary>
    /// The number of current tab-panes (= 1 if all views are in a single tab, = 2 if there is one splitter etc..)
    /// </summary>
    int PaneCount { get;}
    
    /// <summary>
    /// Get the milliseconds since application start.
    /// </summary>
    int Ticks { get;}
    
    /// <summary>
    /// Get the current time / date as string in the format YYYY-MM-DD hh:mm:ss.lll.
    /// </summary>
    string TimeDateString { get;}
    
    /// <summary>
    /// Get the current time as string in the format hh:mm:ss.lll.
    /// </summary>
    string TimeString { get;}
    
    /// <summary>
    /// Get the loxx version as string (e.g. 1.2.1229).
    /// </summary>
    string Version { get;}
    
    /// <summary>
    /// The number of opened views.
    /// </summary>
    int ViewCount { get;}
    
    /// <summary>
    /// Arrange the current windows with the given mode.
    /// </summary>
    /// <param name="Mode">
    /// The mode how the windows shall be aligned (see enumeration type for possible values).
    /// </param>
    void ArrangeWindows(ArrangeMode Mode);
    
    /// <summary>
    /// Clears the macro output window.
    /// </summary>
    void ClearMacroOutput();
    
    void ClearRecentSettings(string filename);
    
    /// <summary>
    /// Close all open views. The operation will fail if an unsaved view is detected.
    /// </summary>
    void CloseAll();
    
    /// <summary>
    /// Compares the two given files and return true if the files are equal. 
    /// If one of the files is inaccessible an error is returned.
    /// </summary>
    /// <param name="FirstFile">
    /// Path to the first file.
    /// </param>
    /// <param name="SecondFile">
    /// Path to the second file.
    /// </param>
    bool CompareFiles(string FirstFile, string SecondFile);
    
    /// <summary>
    /// Concatenates the given files into a single file.
    /// </summary>
    /// <param name="TargetFile">
    /// Output file of the merge operation.
    /// </param>
    /// <param name="InputFiles">
    /// Input files to be merged.
    /// </param>
    void ConcatFiles(string TargetFile, string[] InputFiles);
    
    /// <summary>
    /// Returns the expression found between the first match of FirstExpression and the first match of SecondExpression. Both expressions
    /// are evaluated as regular expressions.If the given expressions don't match, an empty string is returned.Use case can be a
    /// script extracting values from the line content for further processing.
    /// </summary>
    /// <param name="Value">
    /// The string value, where the search will be performed.
    /// </param>
    /// <param name="FirstExpression">
    /// The first of the 2 strings surrounding the desired expression.
    /// </param>
    /// <param name="SecondExpression">
    /// The second of the 2 strings surrounding the desired expression. If the value is null or empty, the full string after &quot;FirstExpression&quot;
    /// will be returned.
    /// </param>
    string ExtractValue(string Value, string FirstExpression, string SecondExpression);
    
    /// <summary>
    /// Returns the expression found between the first match of FirstExpression and the first match of SecondExpression. Both expressions
    /// are evaluated as regular expressions.If the given expressions don't match, an empty string is returned.Use case can be a
    /// script extracting values from the line content for further processing.
    /// </summary>
    /// <param name="Value">
    /// The string value, where the search will be performed.
    /// </param>
    /// <param name="FirstExpression">
    /// The first of the 2 strings surrounding the desired expression.
    /// </param>
    string ExtractValue(string Value, string FirstExpression);
    
    /// <summary>
    /// Searches for the given expression in the defined set of files.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="fileMask">
    /// Defines the find location. The mask can contain multiple entries separated by semicolon. Example:c:\myfiles\*.cpp;*.hwill
    /// perform the search in *.cpp file and *.h files.The following strings can be used instead of file mask:-- &quot;OpenFiles&quot;:
    /// All open files will be target of the search-- &quot;LastResults&quot;: All files contained in the last find results will
    /// be target of the search
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 intern patterns).
    /// </param>
    /// <param name="countOnly">
    /// If true matching results are only counted.The number of matches for each file is reported in the output window.
    /// </param>
    /// <param name="matchWords">
    /// If true, only word matches are considered.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    /// <param name="recursive">
    /// If true the search will be performed recursively through all sub folders.
    /// </param>
    /// <param name="wait">
    /// If true the function will wait for the completed search.
    /// </param>
    void FindInFiles(string expression, string fileMask, SearchType searchType, bool countOnly, bool matchWords, bool matchCase, bool recursive, bool wait);
    
    /// <summary>
    /// Searches for the given expression in the defined set of files.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="fileMask">
    /// Defines the find location. The mask can contain multiple entries separated by semicolon. Example:c:\myfiles\*.cpp;*.hwill
    /// perform the search in *.cpp file and *.h files.The following strings can be used instead of file mask:-- &quot;OpenFiles&quot;:
    /// All open files will be target of the search-- &quot;LastResults&quot;: All files contained in the last find results will
    /// be target of the search
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 intern patterns).
    /// </param>
    /// <param name="countOnly">
    /// If true matching results are only counted.The number of matches for each file is reported in the output window.
    /// </param>
    /// <param name="matchWords">
    /// If true, only word matches are considered.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    /// <param name="recursive">
    /// If true the search will be performed recursively through all sub folders.
    /// </param>
    void FindInFiles(string expression, string fileMask, SearchType searchType, bool countOnly, bool matchWords, bool matchCase, bool recursive);
    
    /// <summary>
    /// Searches for the given expression in the defined set of files.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="fileMask">
    /// Defines the find location. The mask can contain multiple entries separated by semicolon. Example:c:\myfiles\*.cpp;*.hwill
    /// perform the search in *.cpp file and *.h files.The following strings can be used instead of file mask:-- &quot;OpenFiles&quot;:
    /// All open files will be target of the search-- &quot;LastResults&quot;: All files contained in the last find results will
    /// be target of the search
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 intern patterns).
    /// </param>
    /// <param name="countOnly">
    /// If true matching results are only counted.The number of matches for each file is reported in the output window.
    /// </param>
    /// <param name="matchWords">
    /// If true, only word matches are considered.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    void FindInFiles(string expression, string fileMask, SearchType searchType, bool countOnly, bool matchWords, bool matchCase);
    
    /// <summary>
    /// Searches for the given expression in the defined set of files.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="fileMask">
    /// Defines the find location. The mask can contain multiple entries separated by semicolon. Example:c:\myfiles\*.cpp;*.hwill
    /// perform the search in *.cpp file and *.h files.The following strings can be used instead of file mask:-- &quot;OpenFiles&quot;:
    /// All open files will be target of the search-- &quot;LastResults&quot;: All files contained in the last find results will
    /// be target of the search
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 intern patterns).
    /// </param>
    /// <param name="countOnly">
    /// If true matching results are only counted.The number of matches for each file is reported in the output window.
    /// </param>
    /// <param name="matchWords">
    /// If true, only word matches are considered.
    /// </param>
    void FindInFiles(string expression, string fileMask, SearchType searchType, bool countOnly, bool matchWords);
    
    /// <summary>
    /// Searches for the given expression in the defined set of files.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="fileMask">
    /// Defines the find location. The mask can contain multiple entries separated by semicolon. Example:c:\myfiles\*.cpp;*.hwill
    /// perform the search in *.cpp file and *.h files.The following strings can be used instead of file mask:-- &quot;OpenFiles&quot;:
    /// All open files will be target of the search-- &quot;LastResults&quot;: All files contained in the last find results will
    /// be target of the search
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 intern patterns).
    /// </param>
    /// <param name="countOnly">
    /// If true matching results are only counted.The number of matches for each file is reported in the output window.
    /// </param>
    void FindInFiles(string expression, string fileMask, SearchType searchType, bool countOnly);
    
    /// <summary>
    /// Searches for the given expression in the defined set of files.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="fileMask">
    /// Defines the find location. The mask can contain multiple entries separated by semicolon. Example:c:\myfiles\*.cpp;*.hwill
    /// perform the search in *.cpp file and *.h files.The following strings can be used instead of file mask:-- &quot;OpenFiles&quot;:
    /// All open files will be target of the search-- &quot;LastResults&quot;: All files contained in the last find results will
    /// be target of the search
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 intern patterns).
    /// </param>
    void FindInFiles(string expression, string fileMask, SearchType searchType);
    
    /// <summary>
    /// Searches for the given expression in the defined set of files.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="fileMask">
    /// Defines the find location. The mask can contain multiple entries separated by semicolon. Example:c:\myfiles\*.cpp;*.hwill
    /// perform the search in *.cpp file and *.h files.The following strings can be used instead of file mask:-- &quot;OpenFiles&quot;:
    /// All open files will be target of the search-- &quot;LastResults&quot;: All files contained in the last find results will
    /// be target of the search
    /// </param>
    void FindInFiles(string expression, string fileMask);
    
    /// <summary>
    /// Returns the view with the given (zero based!) index.
    /// </summary>
    /// <param name="Index">
    /// Zero based index.
    /// </param>
    View GetView(int Index);
    
    /// <summary>
    /// Returns the view for the given path (if found). 
    /// If no view is found an error will be returned (check with IsOpen to avoid an error).
    /// </summary>
    /// <param name="Path">
    /// Path of the view to be found.
    /// </param>
    View GetViewByPath(string Path);
    
    /// <summary>
    /// Checks the given string for a regex match.
    /// </summary>
    /// <param name="Value">
    /// The value, where the regex shall be applied.
    /// </param>
    /// <param name="Expression">
    /// The regex expression to be applied.
    /// </param>
    bool IsMatch(string Value, string Expression);
    
    /// <summary>
    /// Returns true if the given path is file behind a running view. Intent is to allow a query for a view without getting an error.
    /// </summary>
    /// <param name="Path">
    /// Path for which a view shall be found.
    /// </param>
    bool IsOpen(string Path);
    
    /// <summary>
    /// Makes a path from the given string. If a relative path is given the current directory is used as base. Environment strings
    /// are replaced and '..' strings are interpreted as 'go up'.Example: MakePath(&quot;%%tmp%%\TestData.txt&quot;) returns
    /// </summary>
    /// <param name="path">
    /// The path that shall be processed.
    /// </param>
    string MakePath(string path);
    
    /// <summary>
    /// Maximizes the main window.
    /// </summary>
    void Maximize();
    
    /// <summary>
    /// Test function for file merging.
    /// </summary>
    /// <param name="TargetFile">
    /// Output file of the merge operation.
    /// </param>
    /// <param name="InputFiles">
    /// Input files to be merged.
    /// </param>
    void MergeFiles(string TargetFile, string[] InputFiles, bool continuousMerge, bool onlyNewData);
    
    /// <summary>
    /// Test function for file merging.
    /// </summary>
    /// <param name="TargetFile">
    /// Output file of the merge operation.
    /// </param>
    /// <param name="InputFiles">
    /// Input files to be merged.
    /// </param>
    void MergeFiles(string TargetFile, string[] InputFiles, bool continuousMerge);
    
    /// <summary>
    /// Test function for file merging.
    /// </summary>
    /// <param name="TargetFile">
    /// Output file of the merge operation.
    /// </param>
    /// <param name="InputFiles">
    /// Input files to be merged.
    /// </param>
    void MergeFiles(string TargetFile, string[] InputFiles);
    
    /// <summary>
    /// Minimizes the main window.
    /// </summary>
    void Minimize();
    
    /// <summary>
    /// Modifies the given file name by inserting a prefix, a postfix or changing the extension.
    /// </summary>
    /// <param name="FileName">
    /// The input file name to be modified.
    /// </param>
    /// <param name="Prefix">
    /// If not empty the file name will be changed into &quot;PrefixName.Extension&quot;
    /// </param>
    /// <param name="Postfix">
    /// If not empty the file name will be changed into &quot;NamePostfix.Extension&quot;
    /// </param>
    /// <param name="Extension">
    /// If not empty the file name will be changed into &quot;Name.Extension.
    /// </param>
    string ModifyFileName(string FileName, string Prefix, string Postfix, string Extension);
    
    /// <summary>
    /// Modifies the given file name by inserting a prefix, a postfix or changing the extension.
    /// </summary>
    /// <param name="FileName">
    /// The input file name to be modified.
    /// </param>
    /// <param name="Prefix">
    /// If not empty the file name will be changed into &quot;PrefixName.Extension&quot;
    /// </param>
    /// <param name="Postfix">
    /// If not empty the file name will be changed into &quot;NamePostfix.Extension&quot;
    /// </param>
    string ModifyFileName(string FileName, string Prefix, string Postfix);
    
    /// <summary>
    /// Modifies the given file name by inserting a prefix, a postfix or changing the extension.
    /// </summary>
    /// <param name="FileName">
    /// The input file name to be modified.
    /// </param>
    /// <param name="Prefix">
    /// If not empty the file name will be changed into &quot;PrefixName.Extension&quot;
    /// </param>
    string ModifyFileName(string FileName, string Prefix);
    
    /// <summary>
    /// Opens the given file and returns a new View object representing the opened view.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    /// <param name="WantsSecondView">
    /// If false (= default), the open call checks whether the file is already opened and if yes, the existing view is activated
    /// and returned.If true new window is opened for the same file.
    /// </param>
    /// <param name="MustExist">
    /// If true the file must exist (function will return with an error if not). If false the file is opened also if it does not
    /// exist.Default value is true.
    /// </param>
    /// <param name="DontWait">
    /// If true the function does not wait for a finished line indexing (can take a long time when opening huge files).
    /// </param>
    View Open(string FilePath, bool WantsSecondView, bool MustExist, bool DontWait);
    
    /// <summary>
    /// Opens the given file and returns a new View object representing the opened view.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    /// <param name="WantsSecondView">
    /// If false (= default), the open call checks whether the file is already opened and if yes, the existing view is activated
    /// and returned.If true new window is opened for the same file.
    /// </param>
    /// <param name="MustExist">
    /// If true the file must exist (function will return with an error if not). If false the file is opened also if it does not
    /// exist.Default value is true.
    /// </param>
    View Open(string FilePath, bool WantsSecondView, bool MustExist);
    
    /// <summary>
    /// Opens the given file and returns a new View object representing the opened view.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    /// <param name="WantsSecondView">
    /// If false (= default), the open call checks whether the file is already opened and if yes, the existing view is activated
    /// and returned.If true new window is opened for the same file.
    /// </param>
    View Open(string FilePath, bool WantsSecondView);
    
    /// <summary>
    /// Opens the given file and returns a new View object representing the opened view.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    View Open(string FilePath);
    
    /// <summary>
    /// Opens the clipboard content as new view.
    /// The content will be copied to the given file (= temp file if not defined).
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// If the file is empty a temp file will be opened.
    /// </param>
    View OpenClipboard(string FilePath);
    
    /// <summary>
    /// Opens the clipboard content as new view.
    /// The content will be copied to the given file (= temp file if not defined).
    /// </summary>
    View OpenClipboard();
    
    /// <summary>
    /// Opens the given file and returns a new View object representing the opened view.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    /// <param name="User">
    /// The user name to be used in credentials.
    /// </param>
    /// <param name="Password">
    /// The password to be used in credentials.
    /// </param>
    /// <param name="AutoUpdate">
    /// If true (= default), the file will perform an auto update of the content with the given period.
    /// </param>
    /// <param name="UpdateInterval">
    /// The update interval (in ms) which shall be used looking for new data.
    /// </param>
    View OpenUrl(string FilePath, string User, string Password, bool AutoUpdate, int UpdateInterval);
    
    /// <summary>
    /// Opens the given file and returns a new View object representing the opened view.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    /// <param name="User">
    /// The user name to be used in credentials.
    /// </param>
    /// <param name="Password">
    /// The password to be used in credentials.
    /// </param>
    /// <param name="AutoUpdate">
    /// If true (= default), the file will perform an auto update of the content with the given period.
    /// </param>
    View OpenUrl(string FilePath, string User, string Password, bool AutoUpdate);
    
    /// <summary>
    /// Opens the given file and returns a new View object representing the opened view.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    /// <param name="User">
    /// The user name to be used in credentials.
    /// </param>
    /// <param name="Password">
    /// The password to be used in credentials.
    /// </param>
    View OpenUrl(string FilePath, string User, string Password);
    
    /// <summary>
    /// Opens the given file and returns a new View object representing the opened view.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    /// <param name="User">
    /// The user name to be used in credentials.
    /// </param>
    View OpenUrl(string FilePath, string User);
    
    /// <summary>
    /// Opens the given file and returns a new View object representing the opened view.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    View OpenUrl(string FilePath);
    
    /// <summary>
    /// Close the application. Unsaved changes are discarded.
    /// </summary>
    void Quit();
    
    /// <summary>
    /// Test only function for resetting the log output (to reduce to relevant parts).
    /// </summary>
    void RestartLogging();
    
    /// <summary>
    /// Rotates the views (same as Ctrl+ Cursor left / right)
    /// </summary>
    /// <param name="left">
    /// if true (= default) the windows rotate to the left.
    /// </param>
    void RotateViews(bool left);
    
    /// <summary>
    /// Rotates the views (same as Ctrl+ Cursor left / right)
    /// </summary>
    void RotateViews();
    
    /// <summary>
    /// Runs the given macro (to be called from extern by name).
    /// </summary>
    /// <param name="Name">
    /// Name of the macro to be executed.
    /// </param>
    void Run(string Name);
    
    /// <summary>
    /// Runs the given macro after a given timeout.
    /// That allows to rerun macros (e.g. if you want to have a cyclic update of a graphical view).
    /// </summary>
    /// <param name="Name">
    /// Name of the macro to be executed.
    /// </param>
    /// <param name="Timeout">
    /// Time in milliseconds for the scheduled start.
    /// </param>
    void ScheduleRun(string Name, uint Timeout);
    
    /// <summary>
    /// Selects the given match (zero based index) in the FindInFiles view.
    /// </summary>
    /// <param name="Index">
    /// Index to be selected.
    /// </param>
    void SelectFindInFilesMatch(int Index);
    
    /// <summary>
    /// Selects a query for the given graphic view.
    /// </summary>
    /// <param name="ViewIndex">
    /// Index of the graphic view where the selection shall be performed.
    /// </param>
    /// <param name="Query">
    /// Name of the query to be selected.
    /// </param>
    bool SelectQuery(int ViewIndex, string Query);
    
    /// <summary>
    /// Sends a ShowWindow message to the main windows and brings it to front.
    /// </summary>
    void Show();
    
    /// <summary>
    /// Display the given graphic data in a graphic output window.
    /// </summary>
    /// <param name="outputIndex">
    /// The target output window (0 == Graphics 1, 1 = graphics 2 etc...)
    /// </param>
    /// <param name="graphicData">
    /// The graphic data to be displayed.
    /// </param>
    void ShowGraph(int outputIndex, GraphData graphicData);
    
    /// <summary>
    /// Sleeps for the given number of milliseconds and dispatches Windows messages. The standard sleep function (System.Threading.Thread.Sleep)
    /// should be avoided - the UI will not update and some states depend on message dispatching.
    /// </summary>
    /// <param name="MilliSeconds">
    /// Time to sleep (milliseconds).
    /// </param>
    void Sleep(int MilliSeconds);
    
    /// <summary>
    /// Splits the given file int equi-sized partial files. The target files are created at the same location as the source file
    /// with an additional _000, _001 etc before the extension.
    /// </summary>
    /// <param name="FileName">
    /// The input file name to be modified.
    /// </param>
    /// <param name="count">
    /// The number of desired target files.
    /// </param>
    void SplitFile(string FileName, int count);
    
    /// <summary>
    /// Starts debug logging to the given file.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the log file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    /// <param name="Level">
    /// The log level (0 = none, 1 = error, 2 = warning, 3 = info)
    /// </param>
    void StartLogging(string FilePath, LogLevel Level);
    
    /// <summary>
    /// Starts debug logging to the given file.
    /// </summary>
    /// <param name="FilePath">
    /// Path to the log file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    void StartLogging(string FilePath);
    
    /// <summary>
    /// Starts given command line
    /// </summary>
    /// <param name="Path">
    /// Name of the macro to be executed.
    /// </param>
    /// <param name="Parameters">
    /// Name of the macro to be executed.
    /// </param>
    void StartProcess(string Path, string Parameters, string CurrentDir, bool Hidden);
    
    /// <summary>
    /// Starts given command line
    /// </summary>
    /// <param name="Path">
    /// Name of the macro to be executed.
    /// </param>
    /// <param name="Parameters">
    /// Name of the macro to be executed.
    /// </param>
    void StartProcess(string Path, string Parameters, string CurrentDir);
    
    /// <summary>
    /// Starts given command line
    /// </summary>
    /// <param name="Path">
    /// Name of the macro to be executed.
    /// </param>
    /// <param name="Parameters">
    /// Name of the macro to be executed.
    /// </param>
    void StartProcess(string Path, string Parameters);
    
    /// <summary>
    /// Starts given command line
    /// </summary>
    /// <param name="Path">
    /// Name of the macro to be executed.
    /// </param>
    void StartProcess(string Path);
    
    /// <summary>
    /// Forces an update of the main window (for repaints while a macro is running).The function is mainly for test purpose only.
    /// </summary>
    void UpdateWindow();
    
    /// <summary>
    /// Writes a message into the macro output window.
    /// </summary>
    /// <param name="Message">
    /// Message string.
    /// </param>
    /// <param name="Type">
    /// Message type for coloring: 0 = Standard1 = Success (green in standard colors)2 = Warning (orange in standard colors))3 =
    /// Error   (red in standard colors)
    /// </param>
    void WriteMessage(string Message, MessageType Type, SourceInfo SourceInfo);
    
    /// <summary>
    /// Writes a message into the macro output window.
    /// </summary>
    /// <param name="Message">
    /// Message string.
    /// </param>
    /// <param name="Type">
    /// Message type for coloring: 0 = Standard1 = Success (green in standard colors)2 = Warning (orange in standard colors))3 =
    /// Error   (red in standard colors)
    /// </param>
    void WriteMessage(string Message, MessageType Type);
    
    /// <summary>
    /// Writes a message into the macro output window.
    /// </summary>
    /// <param name="Message">
    /// Message string.
    /// </param>
    void WriteMessage(string Message);
    
    string queryUserInput(string Title, string DefaultValue);
    
  }
  
  /// <summary>
  /// The view class represents a view to a file and provides a set of functions similar to that what is available in the UI.
  /// </summary>
  public interface View
  {
    /// <summary>
    /// The current focus line.
    /// </summary>
    long CurrentLine { set;get;}
    
    /// <summary>
    /// The content current focus line as string.
    /// </summary>
    string CurrentLineContent { get;}
    
    /// <summary>
    /// The offset of the current focus line in he view.
    /// </summary>
    int CurrentLineOffset { get;}
    
    /// <summary>
    /// If set to true, the file will be deleted when the view is closed. This can be useful for test or intermediate files.
    /// </summary>
    bool DeleteOnClose { set;get;}
    
    /// <summary>
    /// Name of the currently used filter. 
    /// If an unnamed filter is used, the returned name is &quot;&lt; - temp - &gt;&quot;, 
    /// if a quick filter is used the returned value is &quot;&lt; - quick - &gt;&quot; 
    /// and if no filter is selected the returned value is empty (&quot;&quot;);
    /// </summary>
    string Filter { set;get;}
    
    /// <summary>
    /// Returns true as long as a filter is running.
    /// </summary>
    bool FilterRunning { get;}
    
    /// <summary>
    /// The number of find matches.
    /// </summary>
    uint FindMatches { get;}
    
    /// <summary>
    /// Returns true as long as an async find is running.
    /// </summary>
    bool FindRunning { get;}
    
    /// <summary>
    /// Name of the currently used highlight setting. 
    /// If an unnamed setting is used, the returned name is &quot;&lt; - temp - &gt;&quot;, 
    /// and if no setting is selected the returned value is empty (&quot;&quot;);
    /// </summary>
    string Highlighting { set;get;}
    
    /// <summary>
    /// The identifier for this view.
    /// </summary>
    string ID { get;}
    
    /// <summary>
    /// Returns true as long as the index thread is running.
    /// </summary>
    bool IndexRunning { get;}
    
    /// <summary>
    /// The last search time in seconds. This is mainly used in performance tests.
    /// </summary>
    double LastSearchTime { get;}
    
    /// <summary>
    /// Returns true if the line positions are valid (false while line parsing is not completed).
    /// </summary>
    bool LinePositionsValid { get;}
    
    /// <summary>
    /// The number of lines in the view.
    /// </summary>
    long Lines { get;}
    
    /// <summary>
    /// True if all found timestamps are monotone increasing, false if a negative delta time was detected,
    /// </summary>
    bool MonotoneTimestamps { get;}
    
    /// <summary>
    /// The full file name (includes extension) of the file in the view.
    /// </summary>
    string Name { get;}
    
    /// <summary>
    /// The index of the pane containing the view (panes are enumerate from top to bottom, if top positions are equal from left to
    /// right).
    /// </summary>
    int PaneIndex { get;}
    
    /// <summary>
    /// The full file path of the file in the view. In case of online files the url is returned.
    /// </summary>
    string Path { get;}
    
    /// <summary>
    /// The current quick highlight string.
    /// </summary>
    string QuickHighlightString { set;get;}
    
    /// <summary>
    /// The current vertical scroll offset
    /// </summary>
    long ScrollOffset { set;get;}
    
    /// <summary>
    /// The number of selected lines.
    /// </summary>
    int SelectedLines { get;}
    
    /// <summary>
    /// The file size in bytes (limited to 2GB).
    /// </summary>
    long Size { get;}
    
    /// <summary>
    /// Returns the related snippet view. An error is returned if the related view itself is already a snippet view.
    /// </summary>
    View SnippetView { get;}
    
    /// <summary>
    /// If true all current line changes are sending a sync request to other views.
    /// </summary>
    bool SyncLock { set;get;}
    
    /// <summary>
    /// The tab-index of the view inside the containing pane (most left is 0, next is 1 etc...). 
    /// </summary>
    int TabIndex { get;}
    
    /// <summary>
    /// The file title (= name without extension) of the file in the view.
    /// </summary>
    string Title { get;}
    
    /// <summary>
    /// The number of visible lines in the view.
    /// This value is used in tests (might be useless for users).
    /// </summary>
    int VisibleLines { get;}
    
    /// <summary>
    /// Adds all current bookmarks to the snippets.
    /// </summary>
    void AddBookmarksToSnippets();
    
    /// <summary>
    /// Adds the given filter to the current filter settings.
    /// </summary>
    /// <param name="expression">
    /// The expression that shall be used for filtering.
    /// </param>
    /// <param name="matchCase">
    /// If TRUE the filter works case sensitive.
    /// </param>
    /// <param name="searchType">
    /// The search type of the given string (0 = plain, 1 = regex, 2 = wildcards).
    /// </param>
    void AddFilter(string expression, bool matchCase, SearchType searchType);
    
    /// <summary>
    /// Adds the given filter to the current filter settings.
    /// </summary>
    /// <param name="expression">
    /// The expression that shall be used for filtering.
    /// </param>
    /// <param name="matchCase">
    /// If TRUE the filter works case sensitive.
    /// </param>
    void AddFilter(string expression, bool matchCase);
    
    /// <summary>
    /// Adds the given filter to the current filter settings.
    /// </summary>
    /// <param name="expression">
    /// The expression that shall be used for filtering.
    /// </param>
    void AddFilter(string expression);
    
    /// <summary>
    /// Adds all current find matches to the snippet view.
    /// </summary>
    void AddFindMatchesToSnippets();
    
    /// <summary>
    /// Adds the given line to the snippet view.
    /// </summary>
    void AddLineToSnippets(long line);
    
    /// <summary>
    /// Adds the given lines to the snippet view.
    /// </summary>
    void AddLinesToSnippets(long[] line);
    
    /// <summary>
    /// Adds the current selection to the snippet view.
    /// </summary>
    void AddSelectionToSnippets();
    
    /// <summary>
    /// Adds the given value to the related value output.
    /// </summary>
    /// <param name="value">
    /// The value to be added.
    /// </param>
    void AddValue(OutputValue value);
    
    /// <summary>
    /// Appends the given file to the current.
    /// </summary>
    /// <param name="filePath">
    /// Path to the file to be appended.
    /// </param>
    void AppendFile(string filePath);
    
    /// <summary>
    /// Goes back to the original file (from filtered or changed content).
    /// </summary>
    void BackToOrigin();
    
    /// <summary>
    /// Resets the file content to 'everything new from now'. The already contained content will be skipped in the display, only
    /// new lines are displayed.
    /// </summary>
    void Clear();
    
    /// <summary>
    /// Removes all values from the related Output view.
    /// Function is deprecated -- use RemoveAllValues instead.
    /// </summary>
    void ClearAllValues();
    
    /// <summary>
    /// Clears previously configured filters (see AddFilter). 
    /// </summary>
    void ClearFilters();
    
    /// <summary>
    /// Clears the find results from the view.
    /// </summary>
    void ClearFindResults();
    
    /// <summary>
    /// Clears the current selection (same as 'escape' key from keyboard).
    /// </summary>
    void ClearSelection();
    
    /// <summary>
    /// Clears all currently stored snippets.
    /// </summary>
    void ClearSnippets();
    
    /// <summary>
    /// Clears the given value from the related value output.
    /// </summary>
    /// <param name="value">
    /// The value to be removed.
    /// </param>
    void ClearValue(string value);
    
    /// <summary>
    /// Closes the view. Any changes in the file are discarded.
    /// </summary>
    void Close();
    
    /// <summary>
    /// Closes all other views. Any changes in the files are discarded.
    /// </summary>
    void CloseAllButThis();
    
    /// <summary>
    /// Closes the view and removes the file from disk (for test or temp files).
    /// </summary>
    void CloseAndDelete();
    
    /// <summary>
    /// Copies the current selection to the clipboard.
    /// If the selection is too big (&gt; 50MB) the return value is false and up to 50MB will be copied.
    /// If the complete selection is copied the return value is true.
    /// </summary>
    /// <param name="Add">
    /// If true the clipboard content will be extended by the current selection. 
    /// If false, the clipboard content is replaced (normal Ctrl-C).
    /// </param>
    bool CopySelection(bool Add);
    
    /// <summary>
    /// Copies the current selection to the clipboard.
    /// If the selection is too big (&gt; 50MB) the return value is false and up to 50MB will be copied.
    /// If the complete selection is copied the return value is true.
    /// </summary>
    bool CopySelection();
    
    /// <summary>
    /// Create a value in the Value-output
    /// </summary>
    OutputValue CreateValue();
    
    /// <summary>
    /// Removes the current selection from the file.The resulting file will be a temporary file until a save command is performed.
    /// </summary>
    void DeleteSelection();
    
    /// <summary>
    /// Creates a second view to the same file and returns the newly created %classLink(View)% object. 
    /// </summary>
    View Duplicate();
    
    /// <summary>
    /// Opens the file in the editor (same as in UI-context menu).
    /// </summary>
    void Edit();
    
    /// <summary>
    /// Applies a quick filter with the expression (plaint text only). 
    /// To reset the filter say &quot;BackToOrigin&quot;
    /// </summary>
    /// <param name="Expression">
    /// The expression to be used in the filter.
    /// If several remove filters are performed after each other all filters will be applied. 
    /// The filters are reset by a BackToOrigin call.
    /// </param>
    /// <param name="Remove">
    /// If true all matches are removed / if false matches only remain.
    /// </param>
    /// <param name="Wait">
    /// If true the functions returns when the filter is finished.
    /// </param>
    void FilterContent(string Expression, bool Remove, bool Wait);
    
    /// <summary>
    /// Applies a quick filter with the expression (plaint text only). 
    /// To reset the filter say &quot;BackToOrigin&quot;
    /// </summary>
    /// <param name="Expression">
    /// The expression to be used in the filter.
    /// If several remove filters are performed after each other all filters will be applied. 
    /// The filters are reset by a BackToOrigin call.
    /// </param>
    /// <param name="Remove">
    /// If true all matches are removed / if false matches only remain.
    /// </param>
    void FilterContent(string Expression, bool Remove);
    
    /// <summary>
    /// Apply the given array of filters.
    /// </summary>
    /// <param name="Filters">
    /// The filters that shall by applied.
    /// </param>
    /// <param name="Remove">
    /// If true all matches are removed / if false matches only remain.
    /// </param>
    /// <param name="Wait">
    /// If true the function waits for finished filter operation
    /// </param>
    void FilterContentEx(FilterSetting[] Filters, bool Remove, bool Wait);
    
    /// <summary>
    /// Apply the given array of filters.
    /// </summary>
    /// <param name="Filters">
    /// The filters that shall by applied.
    /// </param>
    /// <param name="Remove">
    /// If true all matches are removed / if false matches only remain.
    /// </param>
    void FilterContentEx(FilterSetting[] Filters, bool Remove);
    
    /// <summary>
    /// Searches for the given expression starting at the current line. The operation waits for a finished search (can take a while
    /// in huge files).Return value is the number of matches.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 = simple patterns).
    /// </param>
    /// <param name="collectType">
    /// The collect type for the search (see enumeration for details). All found values can be accessed through the FindMatchInfo
    /// class after a search. 
    /// </param>
    /// <param name="suppressPositionChange">
    /// If set to true the file position does not change caused by a find match.
    /// </param>
    uint Find(string expression, bool matchCase, SearchType searchType, CollectTarget collectType, bool suppressPositionChange);
    
    /// <summary>
    /// Searches for the given expression starting at the current line. The operation waits for a finished search (can take a while
    /// in huge files).Return value is the number of matches.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 = simple patterns).
    /// </param>
    /// <param name="collectType">
    /// The collect type for the search (see enumeration for details). All found values can be accessed through the FindMatchInfo
    /// class after a search. 
    /// </param>
    uint Find(string expression, bool matchCase, SearchType searchType, CollectTarget collectType);
    
    /// <summary>
    /// Searches for the given expression starting at the current line. The operation waits for a finished search (can take a while
    /// in huge files).Return value is the number of matches.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 = simple patterns).
    /// </param>
    uint Find(string expression, bool matchCase, SearchType searchType);
    
    /// <summary>
    /// Searches for the given expression starting at the current line. The operation waits for a finished search (can take a while
    /// in huge files).Return value is the number of matches.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    uint Find(string expression, bool matchCase);
    
    /// <summary>
    /// Searches for the given expression starting at the current line. The operation waits for a finished search (can take a while
    /// in huge files).Return value is the number of matches.
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for.
    /// </param>
    uint Find(string expression);
    
    /// <summary>
    /// Adds all find matches to the current set of bookmarks.
    /// </summary>
    void FindMatchesToBookmarks();
    
    /// <summary>
    /// Returns the delta time (in microseconds) of the given line to the next preceding line having a time stamp.
    /// In case that no time stamp is available an exception is thrown (error return in VB).
    /// </summary>
    /// <param name="Index">
    /// Line index.
    /// </param>
    long GetDeltaTime(long Index);
    
    /// <summary>
    /// Returns all values of the recently performed search. That can be used for a fast retrieval of matching values. 
    /// </summary>
    /// <param name="patternIndex">
    /// The pattern index of the previously performed MultiFind (not required for a simple find).
    /// </param>
    double[] GetDoubleValues(int patternIndex);
    
    /// <summary>
    /// Returns all values of the recently performed search. That can be used for a fast retrieval of matching values. 
    /// </summary>
    double[] GetDoubleValues();
    
    /// <summary>
    /// Returns the line number of the Find Match with the given index.
    /// </summary>
    /// <param name="index">
    /// Index of the match.
    /// </param>
    long GetFindMatch(uint index);
    
    /// <summary>
    /// Returns the find match information for the given index.
    /// </summary>
    /// <param name="index">
    /// Index of the match.
    /// </param>
    FindMatchInfo GetFindMatchInfo(uint index);
    
    /// <summary>
    /// Returns the content of the line with the given index.
    /// </summary>
    /// <param name="Line">
    /// Index of the line to be returned.
    /// </param>
    string GetLineContent(long Line);
    
    /// <summary>
    /// Returns the number of matches for the given (zero based) multi-find index.
    /// </summary>
    /// <param name="index">
    /// Pattern index of the previously performed multi-find.
    /// </param>
    uint GetMatchCount(uint index);
    
    /// <summary>
    /// Returns the absolute index of the selected line with the given index (see SelectedLines property returning the number of
    /// selected lines).
    /// </summary>
    /// <param name="Index">
    /// Index of the selected line to be returned.
    /// </param>
    long GetSelectedLine(int Index);
    
    /// <summary>
    /// Returns an array of value pairs containing a string and a count after find operations.A preceding find with value extraction
    /// must be performed.
    /// </summary>
    DecoratedValue[] GetStringCountValues();
    
    /// <summary>
    /// Returns the time (in microseconds) of the given line if available. 
    /// The function returns a negative value if no time is available.
    /// The returned time depends on the date time format in the file, so if a full date (year, month, day) is available the time
    /// bases on January 1, 1601, if no year is in the time stamp it bases on January 1 of the current year.
    /// </summary>
    /// <param name="Index">
    /// Line index.
    /// </param>
    long GetTime(long Index);
    
    /// <summary>
    /// Returns all time stamp values of the recently performed search.That can be used for a fast retrieval of matching x-values.
    /// </summary>
    /// <param name="patternIndex">
    /// The pattern index of the previously performed MultiFind (not required for a simple find).
    /// </param>
    double[] GetTimeStampValues(int patternIndex);
    
    /// <summary>
    /// Returns all time stamp values of the recently performed search.That can be used for a fast retrieval of matching x-values.
    /// </summary>
    double[] GetTimeStampValues();
    
    /// <summary>
    /// Inverts the current selection.
    /// </summary>
    void InvertSelection();
    
    /// <summary>
    /// Reduces the displayed content to the current selection. 
    /// </summary>
    void KeepSelectionOnly();
    
    /// <summary>
    /// Modify a file content. 
    /// Input is an array of replacement blocks with a start/end line and an array of strings to be inserted. 
    /// So a Modification with startLine = 2, endLine = 4, and Lines[] = [&quot;Hello world&quot;] will replace the lines 2 and 3
    /// by the lines &quot;Hello&quot; and &quot;World&quot;.
    /// If no replacement lines are given a block will be only erased. 
    /// If start == end no cut will be performed (insert only).
    /// </summary>
    /// <param name="changes">
    /// If true the function returns before line positions are valid.
    /// </param>
    /// <param name="wait">
    /// If true the operation waits for a finished reload of the changed file.
    /// </param>
    void Modify(Modification[] changes, bool wait);
    
    /// <summary>
    /// Modify a file content. 
    /// Input is an array of replacement blocks with a start/end line and an array of strings to be inserted. 
    /// So a Modification with startLine = 2, endLine = 4, and Lines[] = [&quot;Hello world&quot;] will replace the lines 2 and 3
    /// by the lines &quot;Hello&quot; and &quot;World&quot;.
    /// If no replacement lines are given a block will be only erased. 
    /// If start == end no cut will be performed (insert only).
    /// </summary>
    /// <param name="changes">
    /// If true the function returns before line positions are valid.
    /// </param>
    void Modify(Modification[] changes);
    
    /// <summary>
    /// Moves the view to the given position (see parameters). 
    /// </summary>
    /// <param name="targetPane">
    /// The target pane index. The pane index is counted from top to down (by position) and left to right (in case of identical y-positions).
    /// If the pane index does not exist (too high) it is decremented to the last valid number (no of panes - 1).
    /// </param>
    /// <param name="targetView">
    /// Alternatively to the the target pane index an existing view can be used. 
    /// If a view is given, the targetPane parameter is ignored (the pane is calculated from the view).
    /// If the targetPane value shall be used set this parameter to null.
    /// </param>
    /// <param name="dockMode">
    /// The target dock mode (see enumeration for details).
    /// </param>
    /// <param name="targetTabIndex">
    /// The target tab index if dockMode == Tabbed. 
    /// </param>
    void Move(int targetPane, View targetView, DockMode dockMode, int targetTabIndex);
    
    /// <summary>
    /// Searches for the given expressions starting at the current line. The operation waits for a finished search (can take a while
    /// in huge files).Return value is the number of matches.
    /// </summary>
    /// <param name="expressions">
    /// Expressions to be searched for.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 = simple patterns).
    /// </param>
    /// <param name="collectType">
    /// The collect type for the search (see enumeration for details). All found values can be accessed through the FindMatchInfo
    /// class after a search. 
    /// </param>
    /// <param name="suppressPositionChange">
    /// If set to true the file position does not change caused by a find match.
    /// </param>
    uint MultiFind(string[] expressions, bool matchCase, SearchType searchType, CollectTarget collectType, bool suppressPositionChange);
    
    /// <summary>
    /// Searches for the given expressions starting at the current line. The operation waits for a finished search (can take a while
    /// in huge files).Return value is the number of matches.
    /// </summary>
    /// <param name="expressions">
    /// Expressions to be searched for.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 = simple patterns).
    /// </param>
    /// <param name="collectType">
    /// The collect type for the search (see enumeration for details). All found values can be accessed through the FindMatchInfo
    /// class after a search. 
    /// </param>
    uint MultiFind(string[] expressions, bool matchCase, SearchType searchType, CollectTarget collectType);
    
    /// <summary>
    /// Searches for the given expressions starting at the current line. The operation waits for a finished search (can take a while
    /// in huge files).Return value is the number of matches.
    /// </summary>
    /// <param name="expressions">
    /// Expressions to be searched for.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards, 3 = simple patterns).
    /// </param>
    uint MultiFind(string[] expressions, bool matchCase, SearchType searchType);
    
    /// <summary>
    /// Searches for the given expressions starting at the current line. The operation waits for a finished search (can take a while
    /// in huge files).Return value is the number of matches.
    /// </summary>
    /// <param name="expressions">
    /// Expressions to be searched for.
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    uint MultiFind(string[] expressions, bool matchCase);
    
    /// <summary>
    /// Searches for the given expressions starting at the current line. The operation waits for a finished search (can take a while
    /// in huge files).Return value is the number of matches.
    /// </summary>
    /// <param name="expressions">
    /// Expressions to be searched for.
    /// </param>
    uint MultiFind(string[] expressions);
    
    /// <summary>
    /// Opens the given file (replace the current file in the view).
    /// </summary>
    /// <param name="FilePath">
    /// Path to the file to be opened. The path can contain environment variables (like %%tmp%%).
    /// </param>
    void Open(string FilePath);
    
    /// <summary>
    /// Removes all values from the related Output view.
    /// </summary>
    void RemoveAllValues();
    
    /// <summary>
    /// Clears the given value from the related value output.
    /// </summary>
    /// <param name="value">
    /// The value to be removed.
    /// </param>
    void RemoveValue(string value);
    
    /// <summary>
    /// Resumes the display to 'full file' (opposite operation to 'Clear').
    /// </summary>
    void Resume();
    
    /// <summary>
    /// Saves the file using the original file name.
    /// </summary>
    void Save();
    
    /// <summary>
    /// Saves the file using the given new file name.If the file already exists, the existing file will be overwritten.
    /// </summary>
    /// <param name="FileName">
    /// New target file name.
    /// </param>
    void SaveAs(string FileName);
    
    /// <summary>
    /// Saves a copy of the file using the given new file name.
    /// If the file already exists, the existing file will be overwritten.
    /// </summary>
    /// <param name="FileName">
    /// Target file name for the copy.
    /// </param>
    /// <param name="Open">
    /// If true a new view for the target file will be opened.
    /// </param>
    View SaveCopyAs(string FileName, bool Open);
    
    /// <summary>
    /// Saves a copy of the file using the given new file name.
    /// If the file already exists, the existing file will be overwritten.
    /// </summary>
    /// <param name="FileName">
    /// Target file name for the copy.
    /// </param>
    View SaveCopyAs(string FileName);
    
    /// <summary>
    /// Saves the current selection using the given file name and open a new view to the file.Return value is the newly created %classLink(View)%
    /// object. 
    /// </summary>
    /// <param name="FileName">
    /// Target file name.
    /// </param>
    /// <param name="OverwriteExisting">
    /// If true an existing file will be overwritten. 
    /// </param>
    View SaveSelectionAs(string FileName, bool OverwriteExisting);
    
    /// <summary>
    /// Saves the current selection using the given file name and open a new view to the file.Return value is the newly created %classLink(View)%
    /// object. 
    /// </summary>
    /// <param name="FileName">
    /// Target file name.
    /// </param>
    View SaveSelectionAs(string FileName);
    
    /// <summary>
    /// Performs a selection like a 'space' key on a line in the UI.Depending on the additional parameters an existing selection
    /// is extended / a new selection is added.
    /// </summary>
    /// <param name="line">
    /// The (end) line to select. 
    /// </param>
    /// <param name="extend">
    /// The area shall be extended from the last selection line (like held Shift-key with mouse of space key).
    /// </param>
    /// <param name="add">
    /// If true an existing selection will not be cleared (like held Ctrl-key with mouse of space key).
    /// </param>
    void Select(long line, bool extend, bool add);
    
    /// <summary>
    /// Performs a selection like a 'space' key on a line in the UI.Depending on the additional parameters an existing selection
    /// is extended / a new selection is added.
    /// </summary>
    /// <param name="line">
    /// The (end) line to select. 
    /// </param>
    /// <param name="extend">
    /// The area shall be extended from the last selection line (like held Shift-key with mouse of space key).
    /// </param>
    void Select(long line, bool extend);
    
    /// <summary>
    /// Performs a selection like a 'space' key on a line in the UI.Depending on the additional parameters an existing selection
    /// is extended / a new selection is added.
    /// </summary>
    /// <param name="line">
    /// The (end) line to select. 
    /// </param>
    void Select(long line);
    
    /// <summary>
    /// Selects all lines in the view.
    /// </summary>
    void SelectAll();
    
    /// <summary>
    /// Adds the current find matches to the current selection.
    /// </summary>
    /// <param name="LinesBefore">
    /// Additional lines before the find match to add.
    /// </param>
    /// <param name="LinesAfter">
    /// Additional lines after the find match to add.
    /// </param>
    void SelectFindMatches(int LinesBefore, int LinesAfter);
    
    /// <summary>
    /// Adds the current find matches to the current selection.
    /// </summary>
    /// <param name="LinesBefore">
    /// Additional lines before the find match to add.
    /// </param>
    void SelectFindMatches(int LinesBefore);
    
    /// <summary>
    /// Adds the current find matches to the current selection.
    /// </summary>
    void SelectFindMatches();
    
    /// <summary>
    /// Sends a sync position to other views.
    /// </summary>
    void SendPositionSync();
    
    /// <summary>
    /// Splits the current file the given number of files. 
    /// </summary>
    string[] SplitFile(bool UseTempDir, int Count);
    
    /// <summary>
    /// Splits the file content into threads.
    /// </summary>
    /// <param name="OpenFiles">
    /// If true the files are opened after splitting.
    /// </param>
    void SplitThreads(bool OpenFiles);
    
    /// <summary>
    /// Splits the file content into threads.
    /// </summary>
    void SplitThreads();
    
    /// <summary>
    /// Starts a find operation with the given parameters. The function does not wait for the find to be finished (see Find() for
    /// the synchronous implementation).
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for (see following parameters).
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards. 3 = simple patterns).
    /// </param>
    /// <param name="collectType">
    /// The collect type for the search (see enumeration for details). All found values can be accessed through the FindMatchInfo
    /// class after a search. 
    /// </param>
    void StartFind(string expression, bool matchCase, SearchType searchType, CollectTarget collectType);
    
    /// <summary>
    /// Starts a find operation with the given parameters. The function does not wait for the find to be finished (see Find() for
    /// the synchronous implementation).
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for (see following parameters).
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    /// <param name="searchType">
    /// Search type (0 = plain text, 1 = regular expression, 2 = wild cards. 3 = simple patterns).
    /// </param>
    void StartFind(string expression, bool matchCase, SearchType searchType);
    
    /// <summary>
    /// Starts a find operation with the given parameters. The function does not wait for the find to be finished (see Find() for
    /// the synchronous implementation).
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for (see following parameters).
    /// </param>
    /// <param name="matchCase">
    /// If true a case sensitive search is performed.
    /// </param>
    void StartFind(string expression, bool matchCase);
    
    /// <summary>
    /// Starts a find operation with the given parameters. The function does not wait for the find to be finished (see Find() for
    /// the synchronous implementation).
    /// </summary>
    /// <param name="expression">
    /// Expression to be searched for (see following parameters).
    /// </param>
    void StartFind(string expression);
    
    /// <summary>
    /// Stops a running find operation.
    /// </summary>
    void StopFind();
    
    /// <summary>
    /// Force a sync line of the current line with a related view showing the entire file. 
    /// The function is mainly for test purposes but may be there is also a use case for macros.
    /// </summary>
    void SyncBlock();
    
    /// <summary>
    /// Toggles the bookmark at the current line (sets if unset, clears if set).
    /// </summary>
    void ToggleBookmark();
    
    /// <summary>
    /// Waits for a valid line index (for huge files it can take a while until the line index is valid).
    /// </summary>
    void WaitForValidLines();
    
  }
  
  /// <summary>
  /// Class for configuring a graph. 
  /// The following graphs are supported: 
  /// <para>-- bar charts </para>
  /// <para>-- multi bar charts</para>
  /// <para>-- xy-graphs</para>
  /// </summary>
  public class GraphData
  {
    /// <summary>
    /// For a vertical X-Axis set the value to false (currently not supported).
    /// </summary>
    public bool HorizontalXAxis {get;set;} = true;
    
    /// <summary>
    /// The input data for single bar graphs. 
    /// </summary>
    public MultiBarRow[] MultiBarData {get;set;} = new MultiBarRow[0];
    
    /// <summary>
    /// If true the output in simple bar-charts shows the values
    /// </summary>
    public bool ShowValues {get;set;} = false;
    
    /// <summary>
    /// The input data for single bar graphs. 
    /// </summary>
    public SingleBarRow SingleBarData {get;set;} 
    
    /// <summary>
    /// The diagram title (no title if empty).
    /// </summary>
    public string Title {get;set;} = "";
    
    /// <summary>
    /// Type of the diagram (see enumeration type for possible values).
    /// </summary>
    public GraphType Type {get;set;} = 0;
    
    /// <summary>
    /// Setting for the X-Axis
    /// </summary>
    public Axis XAxis {get;set;} 
    
    /// <summary>
    /// Data for XY-Graphs
    /// </summary>
    public XYRow[] XYGraphData {get;set;} = new XYRow[0];
    
    /// <summary>
    /// Setting for the Y-Axis.
    /// </summary>
    public Axis YAxis {get;set;} 
    
  }
  
  /// <summary>
  /// Class containing a string value and related elements (currently the count of search results)
  /// </summary>
  public class DecoratedValue
  {
    /// <summary>
    /// Number of matches of this value.
    /// </summary>
    public int Count {get;set;} = 0;
    
    /// <summary>
    /// The related string value (found by search).
    /// </summary>
    public string StringValue {get;set;} = "";
    
  }
  
  /// <summary>
  /// Class containing information for a find match
  /// </summary>
  public class FindMatchInfo
  {
    /// <summary>
    /// If collect values was performed with value type &quot;double&quot;, the found value.
    /// </summary>
    public double DoubleValue {get;set;} 
    
    /// <summary>
    /// If collect values was performed with value type double, the found value.
    /// </summary>
    public long IntValue {get;set;} 
    
    /// <summary>
    /// The line where the match is located.
    /// </summary>
    public long Line {get;set;} 
    
    /// <summary>
    /// If a search with value collecting was performed with &quot;collect strings&quot; the matching string value.
    /// </summary>
    public string Match {get;set;} = "";
    
    /// <summary>
    /// The total offset in bytes of the find match in the file.
    /// </summary>
    public long Offset {get;set;} 
    
    /// <summary>
    /// In case of a multi-pattern search the pattern index that matched (0 = first expression, 1 = 2nd etc.).
    /// </summary>
    public int PatternIndex {get;set;} 
    
    /// <summary>
    /// The time stamp of the match in microseconds.If the file includes years, the base is the 01.01.1601 (like Windows FileTime).If
    /// the file does not include years but days and months, the base is the 00:00 at the current years beginning (extra days in
    /// leap years can have an effect here!).
    /// </summary>
    public long TimeStamp {get;set;} 
    
  }
  
  /// <summary>
  /// Class for setting values related to a view. 
  /// </summary>
  public class OutputValue
  {
    /// <summary>
    /// Setting for value alignment in value output, default value is &quot;left&quot;.
    /// </summary>
    public Alignment Align {get;set;} = 0;
    
    /// <summary>
    /// An optional related line (allows a click on the item jumping to the given line)
    /// </summary>
    public long Line {get;set;} = -1;
    
    /// <summary>
    /// A label for the x-axis (e.g. a unit).
    /// </summary>
    public string Name {get;set;} = "";
    
    /// <summary>
    /// If true an icon is displayed for values with Type != Standard.
    /// </summary>
    public bool ShowIcon {get;set;} = false;
    
    /// <summary>
    /// Type of the value (reusing the message type). The coloring and icons depend on the type.
    /// </summary>
    public MessageType Type {get;set;} = 0;
    
    /// <summary>
    /// A label for the y-axis (e.g. a unit).
    /// </summary>
    public string Value {get;set;} = "";
    
  }
  
  /// <summary>
  /// Class containing color information for a line output.
  /// </summary>
  public class ColorInfo
  {
    /// <summary>
    /// Background color. Set to 0xFFFFFFFF if the default value shall be used.
    /// </summary>
    public uint BackColor {get;set;} = 0;
    
    /// <summary>
    /// Foreground color to be used. Set to 0xFFFFFFFF if the default value shall be used.
    /// </summary>
    public uint ForeColor {get;set;} = 0xFFFFFF;
    
    /// <summary>
    /// Length of the color section.
    /// </summary>
    public int Length {get;set;} = -1;
    
    /// <summary>
    /// Start offset of the color (character position).
    /// </summary>
    public int Offset {get;set;} = 0;
    
  }
  
  /// <summary>
  /// Class containing content and color information for a line output.
  /// </summary>
  public class LineFormatInfo
  {
    /// <summary>
    /// An array of colors and positions for highlighting.
    /// </summary>
    public ColorInfo[] Colors {get;set;} = new ColorInfo[0];
    
    /// <summary>
    /// The content to be displayed.
    /// </summary>
    public string Content {get;set;} = "";
    
  }
  
  /// <summary>
  /// Data for a XY graph.
  /// </summary>
  public class XYRow
  {
    /// <summary>
    /// The color for the bar.
    /// </summary>
    public uint Color {get;set;} = 0xFFFFFFFF;
    
    public double Factor {get;set;} = 1.0;
    
    /// <summary>
    /// The (optional) name of the row.
    /// </summary>
    public string Name {get;set;} = "";
    
    /// <summary>
    /// The values of the row.
    /// </summary>
    public XYValue[] Values {get;set;} = new XYValue[0];
    
  }
  
  /// <summary>
  /// A pair of values (x and y) for xy-graphs.
  /// </summary>
  public class XYValue
  {
    /// <summary>
    /// X-Value
    /// </summary>
    public double X {get;set;} 
    
    /// <summary>
    /// Y-Value
    /// </summary>
    public double Y {get;set;} 
    
  }
  
  /// <summary>
  /// A data row for single bar diagrams.
  /// </summary>
  public class SingleBarRow
  {
    /// <summary>
    /// The Values for a bar chart row. 
    /// </summary>
    public SingleBarValue[] Values {get;set;} = new SingleBarValue[0];
    
  }
  
  /// <summary>
  /// Class containing the settings for a diagram axis.
  /// </summary>
  public class Axis
  {
    /// <summary>
    /// If true (= default) the max X-value is calculated automatically from the values.
    /// </summary>
    public bool AutoMax {get;set;} = true;
    
    /// <summary>
    /// If true (= default) the min-value is calculated automatically from the values.
    /// </summary>
    public bool AutoMin {get;set;} = true;
    
    /// <summary>
    /// A label for the axis (e.g. a unit).
    /// </summary>
    public string Label {get;set;} = "";
    
    /// <summary>
    /// The min value of the axis (if AutoMax is false).
    /// </summary>
    public double Max {get;set;} = 0.0;
    
    /// <summary>
    /// The min value of the axis (if AutoMin is false).
    /// </summary>
    public double Min {get;set;} = 0.0;
    
    /// <summary>
    /// In case of a multi-column chart you can add a label for each set of columns (e.g. years on multicolumn diagrams) 
    /// </summary>
    public string[] RowLabels {get;set;} = new string[0];
    
    /// <summary>
    /// If true (= default) the axis shows scalings.
    /// </summary>
    public bool ShowScalings {get;set;} = true;
    
  }
  
  /// <summary>
  /// A data row for a multi-bar diagram.
  /// </summary>
  public class MultiBarRow
  {
    /// <summary>
    /// The color for the values inside the multi-bars.
    /// </summary>
    public uint Color {get;set;} 
    
    /// <summary>
    /// The name of the row.
    /// </summary>
    public string Name {get;set;} = "";
    
    /// <summary>
    /// The Values for a bar chart row. 
    /// </summary>
    public double[] Values {get;set;} = new double[0];
    
  }
  
  /// <summary>
  /// Data for a single bar chart-
  /// </summary>
  public class SingleBarValue
  {
    /// <summary>
    /// The color for the bar.
    /// </summary>
    public uint Color {get;set;} = 0xFFFFFFFF;
    
    /// <summary>
    /// The (optional) label of the value.
    /// </summary>
    public string Label {get;set;} = "";
    
    /// <summary>
    /// The value of this item. 
    /// </summary>
    public double Value {get;set;} 
    
  }
  
  /// <summary>
  /// Loglevel value type.
  /// </summary>
  public class Modification
  {
    /// <summary>
    /// End line of modification block (if same as start line no block will be erased).
    /// </summary>
    public long EndLine {get;set;} = 0;
    
    /// <summary>
    /// Lines to be inserted at the start position. 
    /// </summary>
    public string[] Lines {get;set;} = new string[0];
    
    /// <summary>
    /// Start line of modification block.
    /// </summary>
    public long StartLine {get;set;} = 0;
    
  }
  
  public class FilterSetting
  {
    /// <summary>
    /// The filter expression
    /// </summary>
    public string Expression {get;set;} = "";
    
    /// <summary>
    /// If true the filter is case sensitive (sensitive filters are faster!).
    /// </summary>
    public bool MatchCase {get;set;} 
    
    /// <summary>
    /// The expression type.
    /// </summary>
    public SearchType Type {get;set;} = 0;
    
  }
  
  /// <summary>
  /// Class containing information for a find match.
  /// </summary>
  public class FindInFileMatchInfo
  {
    /// <summary>
    /// The full file path where the match was found.
    /// </summary>
    public string File {get;set;} = "";
    
    /// <summary>
    /// The matching line number.
    /// </summary>
    public long Line {get;set;} 
    
  }
  
  /// <summary>
  /// Class containing information for source infos in the output.
  /// </summary>
  public class SourceInfo
  {
    /// <summary>
    /// The line where the match is located.
    /// </summary>
    public long Line {get;set;} 
    
    /// <summary>
    /// The ID of the related view.
    /// </summary>
    public string ViewID {get;set;} = "";
    
  }
  
  
}